/**
 * 许愿事件：生成物品列表
 * 该代码为lucky_block.ts的辅助函数代码
 * 创建时间：2025-9-4
 * 最后维护时间：2025-9-5
 */

import { Block, ItemStack, Player, system, Vector3, world } from "@minecraft/server"
import { count, ItemList } from "../types/type";
import { randint } from "../types/function";

/**
 * 
 * @param block 许愿方块
 * @param getRedstonePower 红石信号
 * @returns 
 */
export function wishLuckBlock(block: Block, getRedstonePower: number | undefined) {
    // 幸运方块被红石触发（这里的信号必须为15）
    if (getRedstonePower !== 15) return;

    // 获取上面一个方块坐标
    const upPos: Vector3 = {
        x: block.location.x,
        y: block.location.y + 1,
        z: block.location.z
    }

    const dimension = world.getDimension(block.dimension.id)

    // 生成信标
    block.setType('minecraft:beacon')

    // 检查方块类型，如果是水的话就生成水，如果不是的话就生成空气把压力板抵掉
    // 暂时取消这个设定
    // const blockAbove = dimension.getBlock(upPos);
    // if (blockAbove) {
    //     blockAbove.typeId === 'minecraft:water'
    //         ? dimension.setBlockType(upPos, 'minecraft:water')
    //         : dimension.setBlockType(upPos, 'minecraft:air')
    // }

    // 播放音效
    dimension.playSound('beacon.activate', block.location)

    // 放送烟花
    system.run(() => {
        generateFirework(block, {
            x: block.x,
            y: block.y + 1,
            z: block.z
        })
    })

    // 延迟1秒后触发事件
    system.runTimeout(() => {
        // 获取生成的物品id
        const getItemId = wishEventSpawnItem(block)

        // 获取距离许愿井最近的玩家
        const players = dimension.getPlayers({
            location: block.location,
            maxDistance: 10
        });

        // 如果没有找到玩家，则退出
        if (players.length === 0) return

        const player = players[0]
        const item = new ItemStack(getItemId)

        // 发送消息
        sendWishResultMessage(player, item)
    }, 20)
}

// 发送许愿结果消息
function sendWishResultMessage(player: Player, item: ItemStack) {
    if (item.typeId == 'minecraft:tnt') {
        // 使用 '%' 来格式化消息，效果等同于rawtext中的translate
        player.sendMessage(`哦不,是 %${item.localizationKey} ,快跑！`);
        // player.sendMessage({
        //     rawtext: [
        //         { text: '哦不,是 ' },
        //         { translate: item.localizationKey },
        //         { text: ' ,快跑！' }
        //     ]
        // });
    } else {
        player.sendMessage(`许愿成功！你获得了 %${item.localizationKey} ，希望你喜欢它`);
        // player.sendMessage({
        //     rawtext: [
        //         { text: '许愿成功！你获得了 ' },
        //         { translate: item.localizationKey },
        //         { text: ' ，希望你喜欢它' }
        //     ]
        // });
    }
}

// 物品列表为：煤炭，铁锭，金锭，绿宝石，铜锭，紫水晶碎片，钻石，马铃薯，胡萝卜，苹果，猪排，牛排，羊排，TNT

// 物品数量
// 备注：这里用常量表示是因为需要统一控制物品的数量，如果需要单独控制直接修改下面的数组即可，权重同理
const itemCount: count = {
    max: 1,
    min: 5
}
// 物品权重
const weight = 10
// 物品列表
// 备注：概率计算为：(单个普通物品的权重) / (总权重)
const wishEventItemList: ItemList[] = [
    { item: 'minecraft:coal', weight: weight, count: itemCount },
    { item: 'minecraft:iron_ingot', weight: weight, count: itemCount },
    { item: 'minecraft:gold_ingot', weight: weight, count: itemCount },
    { item: 'minecraft:emerald', weight: weight, count: itemCount },
    { item: 'minecraft:copper_ingot', weight: weight, count: itemCount },
    { item: 'minecraft:amethyst_shard', weight: weight, count: itemCount },
    { item: 'minecraft:diamond', weight: weight, count: itemCount },
    { item: 'minecraft:potato', weight: weight, count: itemCount },
    { item: 'minecraft:carrot', weight: weight, count: itemCount },
    { item: 'minecraft:apple', weight: weight, count: itemCount },
    { item: 'minecraft:cooked_porkchop', weight: weight, count: itemCount },
    { item: 'minecraft:cooked_beef', weight: weight, count: itemCount },
    { item: 'minecraft:cooked_mutton', weight: weight, count: itemCount },
    // TNT的权重需要单独调整，如果后续添加新的物品的话可以按需调整它的权重，确保会生成TNT；不建议太少，这样会没有挑战性:)
    { item: 'minecraft:tnt', weight: 20, count: { max: 32, min: 64 } }
]

/**
 * @param block 
 * @returns 物品ID
 * @description 许愿事件：生成物品（包括TNT）
 */
export function wishEventSpawnItem(block: Block): string {
    // 在y+6位置的5×5范围内生成掉落物
    const centerX = block.location.x
    const centerZ = block.location.z
    const dropY = block.location.y + 6

    // 计算总权重
    const totalWeight = wishEventItemList.reduce((sum, item) => sum + item.weight, 0)

    // 基于权重随机选择物品
    let randomValue = Math.random() * totalWeight
    let selectedItem: ItemList | null = null

    for (const item of wishEventItemList) {
        randomValue -= item.weight
        if (randomValue <= 0) {
            selectedItem = item
            break
        }
    }

    // 确保选中了一个物品
    if (!selectedItem) {
        selectedItem = wishEventItemList[0] // 默认选择第一个
    }
    // 随机选择数量
    const randomCount = Math.floor(Math.random() * (selectedItem.count.max - selectedItem.count.min + 1)) + selectedItem.count.min;


    // TNT事件
    if (selectedItem.item === 'minecraft:tnt') {
        // 旧方案
        // 2025-9-30 已废除
        // for (let xOffset = -2; xOffset <= 2; xOffset++) {
        //     for (let zOffset = -2; zOffset <= 2; zOffset++) {
        //         const spawnLocation = {
        //             x: centerX + xOffset + 0.5,
        //             y: dropY,
        //             z: centerZ + zOffset + 0.5
        //         }

        //         // 延迟0.5秒生成，给予逃生机会
        //         system.runTimeout(() => {
        //             world.getDimension(block.dimension.id).spawnEntity(
        //                 'minecraft:tnt',
        //                 spawnLocation
        //             )
        //         }, 10)

        //     }
        // }

        // 生成位置
        const pos = {
            x: block.location.x,
            y: dropY,
            z: block.location.z
        }
        // 生成TNT并给予随机击退，TNT密度由randomCount控制
        // 备注：击退是为了模拟TNT随机生成位置，避免挤在一块
        for (let i = 0; i < randomCount; i++) {
            // 延迟1游戏刻生成，给予逃生机会
            system.runTimeout(() => {
                world.getDimension(block.dimension.id).spawnEntity(
                    'minecraft:tnt',
                    pos
                ).applyImpulse({
                    x: (Math.random() * 2 - 1) * randint(0.1, 0.3, true),
                    y: randint(0.4, 0.8, true),
                    z: (Math.random() * 2 - 1) * randint(0.1, 0.3, true)
                })
            }, 1)
        }
    } else {
        // 其他物品正常生成掉落物
        for (let xOffset = -2; xOffset <= 2; xOffset++) {
            for (let zOffset = -2; zOffset <= 2; zOffset++) {
                const dropLocation = {
                    x: centerX + xOffset + 0.5,
                    y: dropY,
                    z: centerZ + zOffset + 0.5
                };


                // 生成物品
                const item = new ItemStack(selectedItem.item, randomCount)

                world.getDimension(block.dimension.id).spawnItem(
                    item,
                    dropLocation
                );
            }
        }
    }
    return selectedItem.item
}

// 生成烟花辅助函数
function generateFirework(block: Block, location: Vector3): void {
    // 生成5-10个烟花火箭
    const spawnCount = Math.floor(Math.random() * 6) + 5; // 5到10个

    for (let i = 0; i < spawnCount; i++) {
        // 在5×5范围内随机选择位置
        const xOffset = Math.floor(Math.random() * 5) - 2; // -2到2
        const zOffset = Math.floor(Math.random() * 5) - 2; // -2到2

        // 生成烟花火箭位置
        const spawnLocation = {
            x: location.x + xOffset + 0.5,
            y: location.y,
            z: location.z + zOffset + 0.5
        };

        // 生成物品
        world.getDimension(block.dimension.id).spawnEntity(
            'minecraft:fireworks_rocket',
            spawnLocation
        )
    }
}