import { system, CustomCommandStatus, CommandPermissionLevel, world } from '@minecraft/server'
import { debugDrawer, DebugBox } from '@minecraft/debug-utilities'

const trackedEntities = [];

const summonCmd = {
    name: 'debug:tickworld',
    description: "Summons an entity with the tick_world component, teleports it 200 blocks away, and prints whether the chunk it\'s in is loaded every second.",
    permissionLevel: CommandPermissionLevel.Any
};

function summonCmdCallback(origin) {
    const source = origin.sourceEntity;
    const location = source?.location;
    if (location) {
        system.run(() => {
            summonTickWorldEntity(source.dimension, location);
        });
        return { status: CustomCommandStatus.Success, message: "Summoned a new tick_world entity." };
    } else {
        return { status: CustomCommandStatus.Success, message: "Could not summon a new tick_world entity." };
    }
};

function summonTickWorldEntity(dimension, location) {
    const entity = dimension.spawnEntity('debug:tickworld', location, { initialPersistence: true });
    startEntityDisplay(entity);
    system.runTimeout(() => {
        teleportToUnloadedChunks(entity, 200);
    }, 1);
}

function teleportToUnloadedChunks(entity, distance) {
    const newLocation = { x: entity.location.x + distance, y: entity.location.y, z: entity.location.z + distance };
    entity.teleport(newLocation);
}

function startEntityDisplay(entity) {
    const debugShape = new DebugBox(entity.location);
    trackedEntities.push({ entity, debugShape });
    debugDrawer.addShape(debugShape);
}

function cleanupUntrackedEntities() {
    world.getAllPlayers()[0]?.dimension.getEntities({ type: 'debug:tickworld' })
        .filter((entity) => !trackedEntities.map((trackedEntity) => trackedEntity.entity).includes(entity))
        .forEach((entity) => entity.remove());
}

system.beforeEvents.startup.subscribe((event) => {
    event.customCommandRegistry.registerCommand(summonCmd, summonCmdCallback);
});

system.runInterval(() => {
    cleanupUntrackedEntities();
    for (let i = 0; i < trackedEntities.length; i++) {
        const trackedEntity = trackedEntities[i];
        if (!trackedEntity?.entity) return;
        const location = trackedEntity.entity.location;
        trackedEntity.debugShape.location = location;
        world.sendMessage(`Entity ${i}: isLoaded? ${trackedEntity.entity.dimension.isChunkLoaded(location)}, location: [${Math.floor(location.x)}, ${Math.floor(location.y)}, ${Math.floor(location.z)}]`);
    }
}, 20);