import { world, system } from "@minecraft/server";
import { ActionFormData, ModalFormData } from "@minecraft/server-ui";
import * as languages from "./languages";

const langList = Object.keys(languages);
const defaultFieldAssign = {
    1: "direction",
    2: "chunkLoc",
    3: "yaw",
    4: "biome",
    5: "lightLevel",
    6: "elTime",
    7: "dimension",
    8: "entityCount",
    9: "inVillage",
    10: "isUnderground",
    11: "empty",
    12: "weather",
    13: "dayCount",
    14: "moonPhase",
    15: "difficulty",
    16: "tps"
}

// menu register
const menuName = "Debug-Screen";
const menuPersonaId = "4c8ae710-df2e-47cd-814d-cc7bf21a3d67";
const dim = world.getDimension("overworld");
dim.runCommandAsync(`scriptevent "star-menu:${menuName}" ${menuPersonaId}`);

var lastTick = Date.now();
var worldTps = 20;
var timeArray = [];
system.runInterval(() => {
    if (timeArray.length === 20) timeArray.shift();
    timeArray.push(Date.now() - lastTick);
    worldTps = Math.round(1000 / (timeArray.reduce((a, b) => a + b) / timeArray.length));
    lastTick = Date.now();
})

function getElapsedTime(joinTime) {
    const ticks = system.currentTick - joinTime;
    const secs = Math.floor((ticks / 20) % 60).toString().padStart(2, "0");
    const mins = Math.floor((ticks / 1200) % 60).toString().padStart(2, "0");
    const hours = Math.floor(ticks / 72000).toString().padStart(2, "0");
    return hours + ":" + mins + ":" + secs;
}

function getViewDirection(player, lang) {
    var angle = player.getRotation().y;
    if(angle < 0) angle = 360 - Math.abs(angle);
    if((225 < angle) && (angle <= 315)) {
        return lang.directionList.east;
    } else if((45 <= angle) && (angle < 135)) {
        return lang.directionList.west;
    } else if ((135 < angle) && (angle <= 225)) {
        return lang.directionList.north;
    } else {
        return lang.directionList.south;
    }
}

function getChunkLoc(player) {
    const location = player.location;
    const chunkX = Math.floor(location.x/16);
    const chunkZ = Math.floor(location.z/16);
    const locInChunkX = Math.floor(location.x - 16 * chunkX);
    const locInChunkZ = Math.floor(location.z - 16 * chunkZ);
    return `${chunkX}, ${chunkZ} (${locInChunkX}, ${locInChunkZ})`;
}

function getYaw(player) {
    return `${Math.round(player.getRotation().x*(-1))}deg`;
}

function fieldMenuGenerator(player, lang, fieldAssign, fieldList, panel="left") {
    const form = new ModalFormData()
    form.title(lang.menu.fieldTitle);

    const vars = {
        left: {
            i: 1,
            k: 5
        },
        right: {
            i: 6,
            k: 16
        }
    };

    for(var i = vars[panel]["i"]; i <= vars[panel]["k"]; i++) {
        form.dropdown(lang.menu.fieldDropdownLabel+`${i}`, fieldList, fieldList.indexOf(lang.labels[fieldAssign[i]].slice(0, -1)));
    }

    return form.show(player).then((r) => {
        if(r.canceled) return fieldAssign;
        var i = vars[panel]["i"];
        for(var selection in r.formValues) {
            fieldAssign[String(i)] = Object.keys(lang.labels)[r.formValues[selection]];
            i = i+1;
        }
        player.setDynamicProperty("fields", JSON.stringify(fieldAssign));
        player.onScreenDisplay.setActionBar(`§6${lang.menu.fieldUpdateMessage}`);
        return fieldAssign;
    });
}

world.afterEvents.playerSpawn.subscribe((e) => {
    if(e.initialSpawn == false) return;
    const joinTime = system.currentTick;
    const player = e.player;

    // field assignment setup
    if(player.getDynamicProperty("fields") == undefined) {
        player.setDynamicProperty("fields", JSON.stringify(defaultFieldAssign));
    }
    var fieldAssign = JSON.parse(player.getDynamicProperty("fields"));

    // language setup
    if(player.getDynamicProperty("language") == undefined) {
        player.setDynamicProperty("language", "English");
    }
    var lang = languages[player.getDynamicProperty("language")];

    // menu system
    const menuEvent = system.afterEvents.scriptEventReceive.subscribe((e) => {
        if(e.id != "star-menu:trigger" || e.message != "Debug-Screen" || e.sourceEntity != player) return;

        new ActionFormData()
        .title(lang.menu.mainTitle)
        .body(lang.menu.mainBody)
        .button(lang.menu.mainLangBtn)
        .button(lang.menu.mainFieldBtn)
        .show(player)
        .then((r) => {
            if(r.canceled) return;
            else if(r.selection == 0) {
                new ModalFormData()
                .title(lang.menu.langTitle)
                .dropdown(lang.menu.langDropdownLabel, langList, langList.indexOf(player.getDynamicProperty("language")))
                .show(player)
                .then((r) => {
                    if(r.canceled) return;
                    const newLang = langList[r.formValues[0]];
                    player.setDynamicProperty("language", newLang);
                    lang = languages[newLang];
                    player.onScreenDisplay.setActionBar(`§6${lang.menu.langUpdateMessage} §a${newLang}`);
                });
            } else if(r.selection == 1) {
                var fieldList = [];
                Object.keys(lang.labels).forEach((f) => {
                    fieldList.push(lang.labels[f].slice(0, -1))
                });
                new ActionFormData()
                .title(lang.menu.fieldTitle)
                .body(lang.menu.fieldPanelSelectBody)
                .button(lang.menu.leftPanelBtn)
                .button(lang.menu.rightPanelBtn)
                .show(player)
                .then((r) => {
                    if(r.canceled) return;
                    if(r.selection == 0) {
                        fieldMenuGenerator(player, lang, fieldAssign, fieldList, "left")
                        .then((r) => {
                            fieldAssign = r;
                        });
                    } else if(r.selection == 1) {
                        fieldMenuGenerator(player, lang, fieldAssign, fieldList, "right")
                        .then((r) => {
                            fieldAssign = r;
                        });
                    }
                });
            }
        });
    });

    // data storage object
    var data = {
        empty: "",
        biome: "",
        lightLevel: "",
        difficulty: "",
        inVillage: "",
        isUnderground: "",
        weather: "",
        elTime: "",
        chunkLoc: "",
        entityCount: "",
        dimension: "",
        direction: "",
        dayCount: "",
        moonPhase: "",
        yaw: "",
        tps: "",
    }

    // data event recievers
    const dataEvent = world.afterEvents.dataDrivenEntityTrigger.subscribe((e) => {
        if(e.entity != player || !e.eventId.startsWith("star")) return;
        const id = e.eventId.slice(5, 9);
        const value = e.eventId.slice(10);
        
        if(id == "0001") {
            data.lightLevel = lang.labels.lightLevel+" "+value;
        } else if(id == "0010") {
            data.difficulty = lang.labels.difficulty+" "+lang.difficultyList[value];
        } else if(id == "0011") {
            data.inVillage = lang.labels.inVillage+" "+lang.booleans[value];
        } else if(id == "0100") {
            data.isUnderground = lang.labels.isUnderground+" "+lang.booleans[value];
        } else if(id == "0101") {
            data.biome = lang.labels.biome+" "+lang.biomeList[value];
        } else if(id == "0110") {
            data.weather = lang.labels.weather+" "+lang.weatherList[value];
        }
    });
    
    // data transfer
    const runId = system.runInterval(() => {
        data.elTime = lang.labels.elTime+" "+getElapsedTime(joinTime);
        data.direction = lang.labels.direction+" "+getViewDirection(player, lang);
        data.chunkLoc = lang.labels.chunkLoc+" "+getChunkLoc(player);
        data.yaw = lang.labels.yaw+" "+getYaw(player);
        data.entityCount = lang.labels.entityCount+" "+player.dimension.getEntities({location: player.location, maxDistance: 96}).length;
        data.dimension = lang.labels.dimension+" "+lang.dimensionList[player.dimension.id];
        data.dayCount = lang.labels.dayCount+" "+world.getDay();
        data.moonPhase = lang.labels.moonPhase+" "+world.getMoonPhase();
        data.tps = lang.labels.tps+" "+worldTps;

        var dataString = "";
        Object.keys(fieldAssign).forEach((f) => {
            const d = data[fieldAssign[f]].padStart(100, "!");
            dataString = dataString+d;
        });
        player.onScreenDisplay.setTitle(dataString);
    }, 1);

    // setup to stop the all events and callbacks when the player leaves
    const leaveEvent = world.beforeEvents.playerLeave.subscribe((e) => {
        if(e.player != player) return;
        system.clearRun(runId);
        system.run(() => {
            system.afterEvents.scriptEventReceive.unsubscribe(menuEvent);
            world.afterEvents.dataDrivenEntityTrigger.unsubscribe(dataEvent);
            world.beforeEvents.playerLeave.unsubscribe(leaveEvent);
        });
    });
});
