package net.minecraft.block;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.EnumSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;
import java.io.PrintWriter;
import javax.annotation.Nullable;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.BlockFaceShape;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.Mirror;
import net.minecraft.util.Rotation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

public class RedstoneWireTurbo {
    /*
     * This is Helper class for BlockRedstoneWire.  It implements a minimially-invasive
     * bolt-on accelerator that performs a breadth-first search through redstone wire blocks
     * in order to more efficiently and deterministicly compute new redstone wire power levels
     * and determine the order in which other blocks should be updated.  
     *
     * Features:
     * - Changes to BlockRedstoneWire are very limited, no other classes are affected, and the 
     *   choice between old and new redstone wire update algorithms is switchable on-line.
     * - The vanilla implementation relied on World.notifyNeighborsOfStateChange for redstone 
     *   wire blocks to communicate power level changes to each other, generating 36 block 
     *   updates per call.  This improved implementation propagates power level changes directly
     *   between redstone wire blocks.  Redstone wire power levels are therefore computed more quickly, 
     *   and block updates are sent only to non-redstone blocks, many of which may perform an 
     *   action when informed of a change in redstone power level.  (Note:  Block updates are not
     *   the same as state changes to redstone wire.  Wire block states are updated as soon
     *   as they are computed.)
     * - Of the 36 block updates generated by a call to World.notifyNeighborsOfStateChange,
     *   12 of them are obviously redundant (e.g. the west neighbor of the east neighbor).
     *   These are eliminated.
     * - Updates to redstone wire and other connected blocks are propagated in a breath-first
     *   manner, radiating out from the initial trigger (a block update to a redstone wire
     *   from something other than redstone wire).
     * - Updates are scheduled both deterministicly and in an intuitive order, addressing bug
     *   MC-11193. 
     * - All redstone behavior that used to be locational now works the same in all locations.
     * - A few behaviors still depend on orientation; for instance, west is processed before east.
     * - Information that is otherwise computed over and over again or which is expensive to 
     *   to compute is cached for faster lookup.  This includes coordinates of block position 
     *   neighbors and block states that won't change behind our backs during the execution of 
     *   this search algorithm.
     * - Redundant block updates (both to redstone wire and to other blocks) are heavily 
     *   consolidated.  For worst-case scenarios (depowering of redstone wire) this results
     *   in a reduction of block updates by as much as 95% (factor of 1/21).  Due to overheads, 
     *   empirical testing shows a speedup better than 6x.  This addresses bug MC-81098.
     *
     * Extensive testing has been performed to ensure that existing redstone contraptions still
     * behave as expected.  Results of early testing that identified undesirable behavior changes
     * were addressed.  Additionally, real-time performance testing revealed compute inefficiencies
     * With earlier implementations of this accelerator.  Some compatibility adjustments and 
     * performance optimizations resulted in harmless increases in block updates above the 
     * theoretical minimum.
     * 
     * Only a single redstone machine was found to break:  An instant dropper line hack that
     * relies on powered rails and quasiconnectivity but doesn't work in all directions.  The 
     * replacement is to lay redstone wire directly on top of the dropper line, which now works
     * reliably in any direction.
     * 
     * There are numerous other optimization that can be made, but those will be provided later in
     * separate updates.  This version is designed to be minimalistic.
     *
     * Many thanks to the following individuals for their help in testing this functionality:
     * - pokechu22, _MethodZz_, WARBEN, NarcolepticFrog, CommandHelper (nessie), ilmango,
     *   OreoLamp, Xcom6000, tryashtar, RedCMD, Smokey95Dog, EDDxample, Rays Works,
     *   Nodnam, BlockyPlays.
     */
    
    
    /* Reference to BlockRedstoneWire object, which uses this accelerator */
    BlockRedstoneWire wire;
    
    
    /*
     * Implementation:
     *  
     * Blocks are updated in "layers" radiating out from the initial
     * block update that came from a call to BlockRedstoneWire.neighborChanged().
     * All nodes put in Layer N are those with Manattan distance N from the trigger
     * position, reachable through connected redstone wire blocks.
     *
     * Layer 0 represents the trigger block position that was input to neighborChanged.
     * Layer 1 contains the immediate neighbors of that position.
     * Layer N contains the neighbors of blocks in layer N-1, not including
     *    those in previous layers.
     * 
     * Layers enforce an update order that is a function of Manhattan distance
     * from the initial coordinates input to neighborChanged.  The same 
     * coordinates may appear in multiple layers, but redundant updates are minimized.  
     * Block updates are sent layer-by-layer.  If multiple of a block's neighbors experience 
     * redstone wire changes before its layer is processed, then those updates will be merged.
     * If a block's update has been sent, but its neighboring redstone changes
     * after that, then another update will be sent.  This preserves compatibility with
     * machines that rely on zero-tick behavior, except that the new functionality is non-
     * locational.
     *
     * Within each layer, we use an ArrayList to preserve the 
     * cardinal direction ordering (west, east, etc.) so that update
     * behavior reflects the ordering specified in World.notifyNeighborsOfStateChange.
     * UpdateNode.layer is used to avoid duplicates, so that we can avoid using
     * LinkedHashMap, which is slow.
     */
    private final List<List<UpdateNode>> updateLayers = new ArrayList<>();
    
    
    public RedstoneWireTurbo(BlockRedstoneWire wire) {
        this.wire = wire;
    }

    
    /* 
     * Compute neighbors of a block.  When a redstone wire value changes, previously it called
     * World.notifyNeighborsOfStateChange.  That updates immediately neighboring blocks in
     * west, east, down, up, north, south order.  For each of those neighbors, their own
     * neighbors are updated in the same order.  This generates 36 updates, but 12 of them are
     * redundant; for instance the west neighbor of a block's east neighbor.
    */
    public static BlockPos[] computeAllNeighbors(BlockPos pos) {
        int x = pos.getX();
        int y = pos.getY();
        int z = pos.getZ();
        BlockPos[] n = new BlockPos[24];

        // Immediate neighbors, in the same order as 
        // World.notifyNeighborsOfStateChange, etc.:
        // west, east, down, up, north, south
        n[ 0] = new BlockPos(x-1, y  , z  ); 
        n[ 1] = new BlockPos(x+1, y  , z  ); 
        n[ 2] = new BlockPos(x  , y-1, z  ); 
        n[ 3] = new BlockPos(x  , y+1, z  ); 
        n[ 4] = new BlockPos(x  , y  , z-1); 
        n[ 5] = new BlockPos(x  , y  , z+1); 
    
        // Neighbors of neighbors, in the same order,
        // except that duplicates are not included
        n[ 6] = new BlockPos(x-2, y  , z  );  
        n[ 7] = new BlockPos(x-1, y-1, z  );
        n[ 8] = new BlockPos(x-1, y+1, z  );
        n[ 9] = new BlockPos(x-1, y  , z-1);
        n[10] = new BlockPos(x-1, y  , z+1);
        n[11] = new BlockPos(x+2, y  , z  );
        n[12] = new BlockPos(x+1, y-1, z  );
        n[13] = new BlockPos(x+1, y+1, z  );
        n[14] = new BlockPos(x+1, y  , z-1);
        n[15] = new BlockPos(x+1, y  , z+1);
        n[16] = new BlockPos(x  , y-2, z  );
        n[17] = new BlockPos(x  , y-1, z-1);
        n[18] = new BlockPos(x  , y-1, z+1);
        n[19] = new BlockPos(x  , y+2, z  );
        n[20] = new BlockPos(x  , y+1, z-1);
        n[21] = new BlockPos(x  , y+1, z+1);
        n[22] = new BlockPos(x  , y  , z-2);
        n[23] = new BlockPos(x  , y  , z+2);
        return n;
    }

    /*
     * Structure to keep track of redstone wire blocks and
     * neighbors that will receive updates.
     */
    private static class UpdateNode {
        public static enum Type {
            UNKNOWN, REDSTONE, OTHER
        }
        
        IBlockState currentState;       // Keep track of redstone wire value
        UpdateNode[] neighbor_nodes;    // References to neighbors (directed graph edges)
        BlockPos self;                  // UpdateNode's own position
        BlockPos parent;                // Which block pos spawned/updated this node
        Type type = Type.UNKNOWN;       // unknown, redstone wire, other type of block
        int layer;                      // Highest layer this node is scheduled in
    }
    
    
    /*
     * Keep track of all block positions discovered during search and their current states.
     * We want to remember one entry for each position.
     */
    private final Map<BlockPos, UpdateNode> nodeCache = new HashMap<>();



    /*
     * Process a node that is newly created or whose neighboring redstone wire
     * has experienced value changes.
     */
    private void updateNode(World worldIn, UpdateNode upd1, int layer) {
        BlockPos pos = upd1.self;
        IBlockState oldState;

        if (upd1.type == UpdateNode.Type.UNKNOWN) {
            // For a block not visited yet, look up its current state and
            // cache it.
            oldState = worldIn.getBlockState(pos);
            upd1.currentState = oldState;

            // Some neighbors of redstone wire are other kinds of blocks.
            // These need to receive block updates to inform them that
            // redstone wire values have changed.                              
            Block block = oldState.getBlock();
            if (block != wire) {
                // Mark this block as not redstone wire and therefore
                // requiring updates
                upd1.type = UpdateNode.Type.OTHER;
                
                // Non-redstone blocks may propagate updates, but those updates
                // are not handled by this accelerator.  Therefore, we do not
                // expand this position's neighbors.
                return;
            }

            // One job of BlockRedstoneWire.neighborChanged is to convert 
            // redstone wires to items if the block beneath was removed.
            // With this accelerator, BlockRedstoneWire.neighborChanged
            // is only typically called for a single wire block, while
            // others are processed internally by the breadth first search
            // algorithm.  To preserve this game behavior, this check must
            // be replicated here.
            if (!wire.canPlaceBlockAt(worldIn, pos)) {
                // Pop off the redstone dust
                wire.dropBlockAsItem(worldIn, pos, oldState, 0);
                worldIn.setBlockToAir(pos);
                
                // Mark this position as not being redstone wire
                upd1.type = UpdateNode.Type.OTHER;
                
                // Note: Sending updates to air blocks leads to an empty method.
                // Testing shows this to be faster than explicitly avoiding updates to
                // air blocks.
                return;
            }

            // If the above conditions fail, then this is a redstone wire block.
            upd1.type = UpdateNode.Type.REDSTONE;
        } else {
            // For a previously-visited block, look up the last known state.
            // Due to the way other redstone components are updated, we do not
            // have to worry about a state changing behind our backs.  The rare
            // exception is handled by scheduleReentrantNeighborChanged.
            oldState = upd1.currentState;
        }
    
        // Ask the wire block to compute its power level from its neighbors.
        // This will also update the wire's power level and return a new
        // state if it has changed.  When a wire power level is changed, 
        // calculateCurrentChanges will immediately update the block state in the world
        // and return the same value here to be cached in the corresponding
        // UpdateNode object.  
        // Future optimization opportunity:  Looking up block state is slow.  
        // calculateCurrentChanges could look at cached wire values for a
        // significant performance boost.
        IBlockState newState = wire.calculateCurrentChanges(worldIn, pos, pos, oldState);

        // Only inform neighors if the state has changed
        if (newState != oldState) {
            // Store the new state
            upd1.currentState = newState;

            // Inform neighbors of the change
            propagateChanges(worldIn, upd1, layer);
        }
    }

    /*
     * For any redstone wire block in layer N, inform immediate neighbors
     * to recompute in layer N+1, and neighbors of neighbors to recompute
     * in layer N+2.
     */
    private void propagateChanges(World worldIn, UpdateNode upd1, int layer) {
        // Make sure there are enough layers in the list
        while (updateLayers.size() <= layer+2) updateLayers.add(new ArrayList<UpdateNode>());

        BlockPos pos = upd1.self;

        if (upd1.neighbor_nodes == null) {
            // If the node being updated (upd1) is new, expand it so that the
            // search graph includes its neighbors...
            
            // Get the list of neighbor coordinates
            BlockPos[] neighbors = computeAllNeighbors(pos);
            upd1.neighbor_nodes = new UpdateNode[24];

            // Iterate the neigbors
            for (int i=0; i<24; i++) {
                // In the neighbor list, entries 0 to 5 are immediate neighbors, so
                // add them to layer+1.  The remaining entries are neighbors of neighbors,
                // so add to layer+2.
                int layer2 = layer + (i<6 ? 1 : 2);
                
                // Look to see if there is an existing UpdateNode for this neighbor
                BlockPos pos2 = neighbors[i];
                UpdateNode upd2 = nodeCache.get(pos2);

                if (upd2 == null) {
                    // If this neighbor has not been discovered before, create a new
                    // UpdateNode for it.
                    upd2 = new UpdateNode();
                    upd2.self = pos2;
                    upd2.layer = layer2;
     
                    // Store the new node in the cache of known blocks
                    nodeCache.put(pos2, upd2);

                    // Schedule this neighbor to be processed in layer+1 or layer+2
                    updateLayers.get(layer2).add(upd2);
    
                    // Keep track of which block spawned this neighbor
                    upd2.parent = pos;
                } else {
                    // If this neighbor is previously known, we want to minimize redundant
                    // updates.  This test ensures that an update is never scheduled for a
                    // layer that it is already in.
                    if (layer2 > upd2.layer) {
                        // Update highest layer this UpdateNode is in
                        upd2.layer = layer2;

                        // Schedule this neighbor to be processed in layer+1 or layer+2
                        updateLayers.get(layer2).add(upd2);

                        // Keep track of which block updated this neighbor
                        upd2.parent = pos;
                    }
                }
 
                // Store this neighbor in UpdateNode's neighbor array
                upd1.neighbor_nodes[i] = upd2;
            }
        } else {
            // If the node being updated (upd1) has already been expanded, then merely
            // schedule updates to its neighbors.
            for (int i=0; i<24; i++) {
                int layer2 = layer + (i<6 ? 1 : 2);
                UpdateNode upd2 = upd1.neighbor_nodes[i];
                if (layer2 > upd2.layer) {
                    upd2.layer = layer2;
                    updateLayers.get(layer2).add(upd2);

                    // Keep track of which block updated this neighbor
                    upd2.parent = pos;
                }
            }
        }
    }

    // The breadth-first search below will send block updates to blocks
    // that are not redstone wire.  If one of those updates results in 
    // a distant redstone wire getting an update, then this.neighborChanged
    // will get called.  This would be a reentrant call, and 
    // it is necessary to properly integrate those updates into the 
    // on-going search through redstone wire.  Thus, we make the layer
    // currently being processed visible at the object level.
    
    // The current layer being processed by the breadth-first search
    private int currentWalkLayer = 0;

    /*
     * Perform a breadth-first (layer by layer) search through redstone
     * wire blocks, propagating value changes to neighbors in an order
     * that is a function of distance from the initial call to 
     * this.neighborChanged.
     */
    private void breadthFirstWalk(World worldIn) 
    {
        // Loop over all layers
        currentWalkLayer = 1;
        while (currentWalkLayer < updateLayers.size()) {
            // Get the set of blocks in this layer
            List<UpdateNode> thisLayer = updateLayers.get(currentWalkLayer);

            // Loop over all blocks in the layer.  Recall that
            // this is a List, preserving the insertion order of
            // cardinal directions (west then east, etc.).
            for (UpdateNode upd : thisLayer) {
                // If the node is new or is redstone wire, evaluate
                // what kind of block it is, what its redstone power is,
                // and schedule updates to neighbors if its value
                // has changed.  
                if (upd.type != UpdateNode.Type.OTHER) {
                    updateNode(worldIn, upd, currentWalkLayer);
                    // Note:  This may change the type to OTHER, which will
                    // result in further processing below.
                }

                // If this block is not redstone wire, send a block update.
                // Redstone wire blocks get state updates, but they don't
                // need block updates.  Only non-redstone neighbors need updates.
                if (upd.type == UpdateNode.Type.OTHER) {
                    // The following function is called "neighborChanged" in the latest
                    // deobfuscated names.  World.func_190524_a is called from
                    // World.notifyNeighborsOfStateChange, and 
                    // notifyNeighborsOfStateExcept.  We don't use 
                    // World.notifyNeighborsOfStateChange here, since we are
                    // already keeping track of all of the neighbor positions
                    // that need to be updated.  All on its own, handling neighbors 
                    // this way reduces block updates by 1/3 (24 instead of 36).
                    worldIn.func_190524_a(upd.self, wire, upd.parent);
                }
            }

            // Clear all update nodes in this layer
            thisLayer.clear();

            // Move on to the next layer
            currentWalkLayer++;
        }
        currentWalkLayer = 0;
    }


    /*
     * Normally, when Minecraft is computing redstone wire power changes, and a wire power level
     * change sends a block update to a neighboring functional component (e.g. piston, repeater, etc.),
     * those updates are queued.  Only once all redstone wire updates are complete will any component
     * action generate any further block updates to redstone wire.  Instant repeater lines, for instance,
     * will process all wire updates for one redstone line, after which the pistons will zero-tick, 
     * after which the next redstone line performs all of its updates.  Thus, each wire is processed in its
     * own discrete wave.
     *
     * However, there are some corner cases where this pattern breaks, with a proof of concept discovered
     * by Rays Works, which works the same in vanilla.  The scenario is as follows:
     * (1) A redstone wire is conducting a signal.
     * (2) Partway through that wave of updates, a neighbor is updated that causes an update to a completely
     *     separate redstone wire.
     * (3) This results in a call to BlockRedstoneWire.neighborChanged for that other wire, in the middle of 
     *     an already on-going propagation through the first wire.
     *
     * The vanilla code, being depth-first, would end up fully processing the second wire before going back
     * to finish processing the first one.  (Although technically, vanilla has no special concept of "being
     * in the middle" of processing updates to a wire.)  For the breadth-first algorithm, we give this
     * situation special handling, where the updates for the second wire are incorporated into the schedule
     * for the first wire, and then the callstack is allowed to unwind back to the on-going search loop in
     * order to continue processing both the first and second wire in the order of distance from the initial
     * trigger.
     */
    private IBlockState scheduleReentrantNeighborChanged(World worldIn, BlockPos pos, IBlockState state, IBlockState newState)
    {
        // Find or generate a node for this block position
        UpdateNode upd = nodeCache.get(pos);
        if (upd == null) {
            upd = new UpdateNode();
            upd.self = pos;
            upd.parent = pos;
            upd.type = UpdateNode.Type.REDSTONE;
            nodeCache.put(pos, upd);
        }
        upd.currentState = newState;

        // Receiving this block update may mean something in the world changed.
        // Therefore we clear the cached block info about all neighbors of
        // the position receiving the update.
        if (upd.neighbor_nodes != null) {
            for (int i=0; i<24; i++) {
                UpdateNode upd2 = upd.neighbor_nodes[i];
                upd2.type = UpdateNode.Type.UNKNOWN;
                upd2.currentState = null;
            }
        }

        // The block at 'pos' is a redstone wire and has been updated already by calling
        // wire.calculateCurrentChanges, so we don't schedule that.  However, we do need
        // to schedule its neighbors.  By passing the current value of 'currentWalkLayer' to
        // propagateChanges, the immediate neighbors of 'pos' are schedule for layer
        // currentWalkLayer+1, and the neighbors of the neighbors are scheduled for layer
        // currentWalkLayer+2.
        propagateChanges(worldIn, upd, currentWalkLayer);

        // Return here.  The call stack will unwind back to the first call to
        // updateSurroundingRedstone, whereupon the new updates just scheduled will
        // be propagated.  This also facilitates elimination of superfluous and
        // redundant block updates.
        return newState;
    }

    /*
     * New version of pre-existing updateSurroundingRedstone, which is called from
     * wire.updateSurroundingRedstone, which is called from wire.neighborChanged and a 
     * few other methods in BlockRedstoneWire.  This sets off the breadth-first 
     * walk through all redstone dust connected to the initial position triggered.
     */
    public IBlockState updateSurroundingRedstone(World worldIn, BlockPos pos, IBlockState state)
    {
        // Check this block's neighbors and see if its power level needs to change
        IBlockState newState = wire.calculateCurrentChanges(worldIn, pos, pos, state);
        
        // If no change, exit
        if (newState == state) {
            return state;
        }

        // Check to see if this update was received during an on-going breadth first search
        if (currentWalkLayer>0 || nodeCache.size()>0) {
            // As breadthFirstWalk progresses, it sends block updates to neighbors.  Some of those
            // neighbors may affect the world so as to cause yet another redstone wire block to receive
            // an update.  If that happens, we need to integrate those redstone wire updates into the
            // already on-going graph walk being performed by breadthFirstWalk. 
            return scheduleReentrantNeighborChanged(worldIn, pos, state, newState);
        }

        // If there are no on-going walks through redstone wire, then start a new walk.
        // First create a node representing the block at 'pos', and then propagate updates
        // to its neighbors.  As stated above, the call to wire.calculateCurrentChanges
        // already performs the update to the block at 'pos', so it is not added to the schedule.
        UpdateNode upd = new UpdateNode();
        upd.self = pos;
        upd.parent = pos;
        upd.currentState = newState;
        upd.type = UpdateNode.Type.REDSTONE;
        nodeCache.put(pos, upd);
        propagateChanges(worldIn, upd, 0);
    
        // Perform the walk over all directly reachable redstone wire blocks, propagating wire value 
        // updates in a breadth first order out from the initial update received for the block at 'pos'.
        breadthFirstWalk(worldIn);

        // With the whole search completed, clear the list of all known blocks.
        // We do not want to keep around state information that may be changed by other code.
        // In theory, we could cache the neighbor block positions, but that is a separate
        // optimization.
        nodeCache.clear();

        return newState;
    }
}
