package net.minecraft.entity.player;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.mojang.authlib.GameProfile;
import io.netty.buffer.Unpooled;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import javax.annotation.Nullable;
import net.minecraft.block.Block;
import net.minecraft.block.BlockFence;
import net.minecraft.block.BlockFenceGate;
import net.minecraft.block.BlockWall;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.crash.CrashReport;
import net.minecraft.crash.CrashReportCategory;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityList;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.IMerchant;
import net.minecraft.entity.passive.EntityHorse;
import net.minecraft.entity.projectile.EntityArrow;
import net.minecraft.init.Items;
import net.minecraft.init.SoundEvents;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.ContainerChest;
import net.minecraft.inventory.ContainerHorseInventory;
import net.minecraft.inventory.ContainerMerchant;
import net.minecraft.inventory.IContainerListener;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.SlotCrafting;
import net.minecraft.item.Item;
import net.minecraft.item.ItemMapBase;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetHandlerPlayServer;
import net.minecraft.network.Packet;
import net.minecraft.network.PacketBuffer;
import net.minecraft.network.play.client.CPacketClientSettings;
import net.minecraft.network.play.server.SPacketAnimation;
import net.minecraft.network.play.server.SPacketCamera;
import net.minecraft.network.play.server.SPacketChangeGameState;
import net.minecraft.network.play.server.SPacketChat;
import net.minecraft.network.play.server.SPacketChunkData;
import net.minecraft.network.play.server.SPacketCloseWindow;
import net.minecraft.network.play.server.SPacketCombatEvent;
import net.minecraft.network.play.server.SPacketCustomPayload;
import net.minecraft.network.play.server.SPacketDestroyEntities;
import net.minecraft.network.play.server.SPacketEffect;
import net.minecraft.network.play.server.SPacketEntityEffect;
import net.minecraft.network.play.server.SPacketEntityStatus;
import net.minecraft.network.play.server.SPacketOpenWindow;
import net.minecraft.network.play.server.SPacketPlayerAbilities;
import net.minecraft.network.play.server.SPacketRemoveEntityEffect;
import net.minecraft.network.play.server.SPacketResourcePackSend;
import net.minecraft.network.play.server.SPacketSetExperience;
import net.minecraft.network.play.server.SPacketSetSlot;
import net.minecraft.network.play.server.SPacketSignEditorOpen;
import net.minecraft.network.play.server.SPacketSoundEffect;
import net.minecraft.network.play.server.SPacketUpdateHealth;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.network.play.server.SPacketUseBed;
import net.minecraft.network.play.server.SPacketWindowItems;
import net.minecraft.network.play.server.SPacketWindowProperty;
import net.minecraft.potion.PotionEffect;
import net.minecraft.scoreboard.IScoreCriteria;
import net.minecraft.scoreboard.Score;
import net.minecraft.scoreboard.ScoreObjective;
import net.minecraft.scoreboard.Team;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.dedicated.DedicatedServer;
import net.minecraft.server.management.PlayerInteractionManager;
import net.minecraft.server.management.UserListOpsEntry;
import net.minecraft.stats.Achievement;
import net.minecraft.stats.AchievementList;
import net.minecraft.stats.StatBase;
import net.minecraft.stats.StatList;
import net.minecraft.stats.StatisticsManagerServer;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityCommandBlock;
import net.minecraft.tileentity.TileEntitySign;
import net.minecraft.util.CooldownTracker;
import net.minecraft.util.CooldownTrackerServer;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EntityDamageSource;
import net.minecraft.util.EnumHand;
import net.minecraft.util.EnumHandSide;
import net.minecraft.util.JsonSerializableSet;
import net.minecraft.util.ReportedException;
import net.minecraft.util.SoundCategory;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.Style;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.village.MerchantRecipeList;
import net.minecraft.world.GameType;
import net.minecraft.world.IInteractionObject;
import net.minecraft.world.ILockableContainer;
import net.minecraft.world.WorldServer;
import net.minecraft.world.biome.Biome;
import net.minecraft.world.storage.loot.ILootContainer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class EntityPlayerMP extends EntityPlayer implements IContainerListener {
   private static final Logger LOGGER = LogManager.getLogger();
   private String language = "en_US";
   public NetHandlerPlayServer connection;
   public final MinecraftServer mcServer;
   public final PlayerInteractionManager interactionManager;
   public double managedPosX;
   public double managedPosZ;
   private final List<Integer> entityRemoveQueue = Lists.<Integer>newLinkedList();
   private final StatisticsManagerServer statsFile;
   private float lastHealthScore = Float.MIN_VALUE;
   private int lastFoodScore = Integer.MIN_VALUE;
   private int lastAirScore = Integer.MIN_VALUE;
   private int lastArmorScore = Integer.MIN_VALUE;
   private int lastLevelScore = Integer.MIN_VALUE;
   private int lastExperienceScore = Integer.MIN_VALUE;
   private float lastHealth = -1.0E8F;
   private int lastFoodLevel = -99999999;
   private boolean wasHungry = true;
   private int lastExperience = -99999999;
   private int respawnInvulnerabilityTicks = 60;
   private EntityPlayer.EnumChatVisibility chatVisibility;
   private boolean chatColours = true;
   private long playerLastActiveTime = System.currentTimeMillis();
   private Entity spectatingEntity;
   private boolean invulnerableDimensionChange;
   private int currentWindowId;
   public boolean isChangingQuantityOnly;
   public int ping;
   public boolean playerConqueredTheEnd;
   private int chunksSinceLastTick = 0;
   private Queue<SPacketChunkData> chunkQueue;

   public EntityPlayerMP(MinecraftServer server, WorldServer worldIn, GameProfile profile, PlayerInteractionManager interactionManagerIn) {
      super(worldIn, profile);
      interactionManagerIn.thisPlayerMP = this;
      this.interactionManager = interactionManagerIn;
      BlockPos blockpos = worldIn.getSpawnPoint();
      if(!worldIn.provider.getHasNoSky() && worldIn.getWorldInfo().getGameType() != GameType.ADVENTURE) {
         int i = Math.max(0, server.getSpawnRadius(worldIn));
         int j = MathHelper.floor_double(worldIn.getWorldBorder().getClosestDistance((double)blockpos.getX(), (double)blockpos.getZ()));
         if(j < i) {
            i = j;
         }

         if(j <= 1) {
            i = 1;
         }

         blockpos = worldIn.getTopSolidOrLiquidBlock(blockpos.add(this.rand.nextInt(i * 2 + 1) - i, 0, this.rand.nextInt(i * 2 + 1) - i));
      }
      this.chunkQueue = new LinkedList<SPacketChunkData>();
      this.mcServer = server;
      this.statsFile = server.getPlayerList().getPlayerStatsFile(this);
      this.stepHeight = 0.0F;
      this.moveToBlockPosAndAngles(blockpos, 0.0F, 0.0F);

      while(!worldIn.getCollisionBoxes(this, this.getEntityBoundingBox()).isEmpty() && this.posY < 255.0D) {
         this.setPosition(this.posX, this.posY + 1.0D, this.posZ);
      }
   }

   public void readEntityFromNBT(NBTTagCompound compound) {
      super.readEntityFromNBT(compound);
      if(compound.hasKey("playerGameType", 99)) {
         if(this.getServer().getForceGamemode()) {
            this.interactionManager.setGameType(this.getServer().getGameType());
         } else {
            this.interactionManager.setGameType(GameType.getByID(compound.getInteger("playerGameType")));
         }
      }
   }

   public void writeEntityToNBT(NBTTagCompound compound) {
      super.writeEntityToNBT(compound);
      compound.setInteger("playerGameType", this.interactionManager.getGameType().getID());
      Entity entity = this.getLowestRidingEntity();
      if(this.getRidingEntity() != null && entity != this & entity.getRecursivePassengersByType(EntityPlayerMP.class).size() == 1) {
         NBTTagCompound nbttagcompound = new NBTTagCompound();
         NBTTagCompound nbttagcompound1 = new NBTTagCompound();
         entity.writeToNBTOptional(nbttagcompound1);
         nbttagcompound.setUniqueId("Attach", this.getRidingEntity().getUniqueID());
         nbttagcompound.setTag("Entity", nbttagcompound1);
         compound.setTag("RootVehicle", nbttagcompound);
      }
   }

   public void addExperienceLevel(int levels) {
      super.addExperienceLevel(levels);
      this.lastExperience = -1;
   }

   public void removeExperienceLevel(int levels) {
      super.removeExperienceLevel(levels);
      this.lastExperience = -1;
   }

   public void addSelfToInternalCraftingInventory() {
      this.openContainer.addListener(this);
   }

   public void sendEnterCombat() {
      super.sendEnterCombat();
      this.connection.sendPacket(new SPacketCombatEvent(this.getCombatTracker(), SPacketCombatEvent.Event.ENTER_COMBAT));
   }

   public void sendEndCombat() {
      super.sendEndCombat();
      this.connection.sendPacket(new SPacketCombatEvent(this.getCombatTracker(), SPacketCombatEvent.Event.END_COMBAT));
   }

   protected CooldownTracker createCooldownTracker() {
      return new CooldownTrackerServer(this);
   }

   public void onUpdate() {
      this.interactionManager.updateBlockRemoving();
      --this.respawnInvulnerabilityTicks;
      if(this.hurtResistantTime > 0) {
         --this.hurtResistantTime;
      }

      this.openContainer.detectAndSendChanges();
      if(!this.worldObj.isRemote && !this.openContainer.canInteractWith(this)) {
         this.closeScreen();
         this.openContainer = this.inventoryContainer;
      }

      while(!this.entityRemoveQueue.isEmpty()) {
         int i = Math.min(this.entityRemoveQueue.size(), Integer.MAX_VALUE);
         int[] aint = new int[i];
         Iterator<Integer> iterator = this.entityRemoveQueue.iterator();
         int j = 0;

         while(iterator.hasNext() && j < i) {
            aint[j++] = ((Integer)iterator.next()).intValue();
            iterator.remove();
         }

         this.connection.sendPacket(new SPacketDestroyEntities(aint));
      }

      Entity entity = this.getSpectatingEntity();
      if(entity != this) {
         if(entity.isEntityAlive()) {
            this.setPositionAndRotation(entity.posX, entity.posY, entity.posZ, entity.rotationYaw, entity.rotationPitch);
            this.mcServer.getPlayerList().serverUpdateMountedMovingPlayer(this);
            if(this.isSneaking()) {
               this.setSpectatingEntity(this);
            }
         } else {
            this.setSpectatingEntity(this);
         }
      }
   }

   public void onUpdateEntity() {
      try {
         super.onUpdate();

         for(int i = 0; i < this.inventory.getSizeInventory(); ++i) {
            ItemStack itemstack = this.inventory.getStackInSlot(i);
            if(itemstack != null && itemstack.getItem().isMap()) {
               Packet<?> packet = ((ItemMapBase)itemstack.getItem()).createMapDataPacket(itemstack, this.worldObj, this);
               if(packet != null) {
                  this.connection.sendPacket(packet);
               }
            }
         }

         if(this.getHealth() != this.lastHealth || this.lastFoodLevel != this.foodStats.getFoodLevel() || this.foodStats.getSaturationLevel() == 0.0F != this.wasHungry) {
            this.connection.sendPacket(new SPacketUpdateHealth(this.getHealth(), this.foodStats.getFoodLevel(), this.foodStats.getSaturationLevel()));
            this.lastHealth = this.getHealth();
            this.lastFoodLevel = this.foodStats.getFoodLevel();
            this.wasHungry = this.foodStats.getSaturationLevel() == 0.0F;
         }

         if(this.getHealth() + this.getAbsorptionAmount() != this.lastHealthScore) {
            this.lastHealthScore = this.getHealth() + this.getAbsorptionAmount();
            this.updateScorePoints(IScoreCriteria.HEALTH, MathHelper.ceiling_float_int(this.lastHealthScore));
         }

         if(this.foodStats.getFoodLevel() != this.lastFoodScore) {
            this.lastFoodScore = this.foodStats.getFoodLevel();
            this.updateScorePoints(IScoreCriteria.FOOD, MathHelper.ceiling_float_int((float)this.lastFoodScore));
         }

         if(this.getAir() != this.lastAirScore) {
            this.lastAirScore = this.getAir();
            this.updateScorePoints(IScoreCriteria.AIR, MathHelper.ceiling_float_int((float)this.lastAirScore));
         }

         if(this.getTotalArmorValue() != this.lastArmorScore) {
            this.lastArmorScore = this.getTotalArmorValue();
            this.updateScorePoints(IScoreCriteria.ARMOR, MathHelper.ceiling_float_int((float)this.lastArmorScore));
         }

         if(this.experienceTotal != this.lastExperienceScore) {
            this.lastExperienceScore = this.experienceTotal;
            this.updateScorePoints(IScoreCriteria.XP, MathHelper.ceiling_float_int((float)this.lastExperienceScore));
         }

         if(this.experienceLevel != this.lastLevelScore) {
            this.lastLevelScore = this.experienceLevel;
            this.updateScorePoints(IScoreCriteria.LEVEL, MathHelper.ceiling_float_int((float)this.lastLevelScore));
         }

         if(this.experienceTotal != this.lastExperience) {
            this.lastExperience = this.experienceTotal;
            this.connection.sendPacket(new SPacketSetExperience(this.experience, this.experienceTotal, this.experienceLevel));
         }

         if(this.ticksExisted % 20 * 5 == 0 && !this.getStatFile().hasAchievementUnlocked(AchievementList.EXPLORE_ALL_BIOMES)) {
            this.updateBiomesExplored();
         }
         
         // Default value thinking:
         // At 10 chunks render-distance, the world has 441 chunks (21^2).
         // I want the world to load in about 10 seconds, and need the rate in chunks/tick.
         // 441 chunks / 10 sec * 1 sec / 20 t = 441/200 chunks/tick, which rounds up to 3.

         DedicatedServer s = (DedicatedServer) this.mcServer;
         while(chunkQueue.size() > 0 && this.chunksSinceLastTick < s.getIntProperty("chunkrate", 3)) {
        	 SPacketChunkData packet = chunkQueue.poll();
        	 connection.sendPacket(packet);
        	 this.chunksSinceLastTick++;
         }
         this.chunksSinceLastTick = 0;
      } catch (Throwable throwable) {
         CrashReport crashreport = CrashReport.makeCrashReport(throwable, "Ticking player");
         CrashReportCategory crashreportcategory = crashreport.makeCategory("Player being ticked");
         this.addEntityCrashInfo(crashreportcategory);
         throw new ReportedException(crashreport);
      }
   }

   private void updateScorePoints(IScoreCriteria criteria, int points) {
      for(ScoreObjective scoreobjective : this.getWorldScoreboard().getObjectivesFromCriteria(criteria)) {
         Score score = this.getWorldScoreboard().getOrCreateScore(this.getName(), scoreobjective);
         score.setScorePoints(points);
      }
   }

   protected void updateBiomesExplored() {
      Biome biome = this.worldObj.getBiomeGenForCoords(new BlockPos(MathHelper.floor_double(this.posX), 0, MathHelper.floor_double(this.posZ)));
      String s = biome.getBiomeName();
      JsonSerializableSet jsonserializableset = (JsonSerializableSet)this.getStatFile().getProgress(AchievementList.EXPLORE_ALL_BIOMES);
      if(jsonserializableset == null) {
         jsonserializableset = (JsonSerializableSet)this.getStatFile().setProgress(AchievementList.EXPLORE_ALL_BIOMES, new JsonSerializableSet());
      }

      jsonserializableset.add(s);
      if(this.getStatFile().canUnlockAchievement(AchievementList.EXPLORE_ALL_BIOMES) && jsonserializableset.size() >= Biome.EXPLORATION_BIOMES_LIST.size()) {
         Set<Biome> set = Sets.newHashSet(Biome.EXPLORATION_BIOMES_LIST);

         for(String s1 : jsonserializableset) {
            Iterator<Biome> iterator = set.iterator();

            while(iterator.hasNext()) {
               Biome biome1 = (Biome)iterator.next();
               if(biome1.getBiomeName().equals(s1)) {
                  iterator.remove();
               }
            }

            if(set.isEmpty()) {
               break;
            }
         }

         if(set.isEmpty()) {
            this.addStat(AchievementList.EXPLORE_ALL_BIOMES);
         }
      }
   }

   public void onDeath(DamageSource cause) {
      boolean flag = this.worldObj.getGameRules().getBoolean("showDeathMessages");
      this.connection.sendPacket(new SPacketCombatEvent(this.getCombatTracker(), SPacketCombatEvent.Event.ENTITY_DIED, flag));
      if(flag) {
         Team team = this.getTeam();
         if(team != null && team.getDeathMessageVisibility() != Team.EnumVisible.ALWAYS) {
            if(team.getDeathMessageVisibility() == Team.EnumVisible.HIDE_FOR_OTHER_TEAMS) {
               this.mcServer.getPlayerList().sendMessageToAllTeamMembers(this, this.getCombatTracker().getDeathMessage());
            } else if(team.getDeathMessageVisibility() == Team.EnumVisible.HIDE_FOR_OWN_TEAM) {
               this.mcServer.getPlayerList().sendMessageToTeamOrAllPlayers(this, this.getCombatTracker().getDeathMessage());
            }
         } else {
            this.mcServer.getPlayerList().sendChatMsg(this.getCombatTracker().getDeathMessage());
         }
      }

      if(!this.worldObj.getGameRules().getBoolean("keepInventory") && !this.isSpectator()) {
         this.inventory.dropAllItems();
      }

      for(ScoreObjective scoreobjective : this.worldObj.getScoreboard().getObjectivesFromCriteria(IScoreCriteria.DEATH_COUNT)) {
         Score score = this.getWorldScoreboard().getOrCreateScore(this.getName(), scoreobjective);
         score.incrementScore();
      }

      EntityLivingBase entitylivingbase = this.getAttackingEntity();
      if(entitylivingbase != null) {
         EntityList.EntityEggInfo entitylist$entityegginfo = (EntityList.EntityEggInfo)EntityList.ENTITY_EGGS.get(EntityList.getEntityString(entitylivingbase));
         if(entitylist$entityegginfo != null) {
            this.addStat(entitylist$entityegginfo.entityKilledByStat);
         }

         entitylivingbase.addToPlayerScore(this, this.scoreValue);
      }

      this.addStat(StatList.DEATHS);
      this.takeStat(StatList.TIME_SINCE_DEATH);
      this.getCombatTracker().reset();
   }

   public boolean attackEntityFrom(DamageSource source, float amount) {
      if(this.isEntityInvulnerable(source)) {
         return false;
      } else {
         boolean flag = this.mcServer.isDedicatedServer() && this.canPlayersAttack() && "fall".equals(source.damageType);
         if(!flag && this.respawnInvulnerabilityTicks > 0 && source != DamageSource.outOfWorld) {
            return false;
         } else {
            if(source instanceof EntityDamageSource) {
               Entity entity = source.getEntity();
               if(entity instanceof EntityPlayer && !this.canAttackPlayer((EntityPlayer)entity)) {
                  return false;
               }

               if(entity instanceof EntityArrow) {
                  EntityArrow entityarrow = (EntityArrow)entity;
                  if(entityarrow.shootingEntity instanceof EntityPlayer && !this.canAttackPlayer((EntityPlayer)entityarrow.shootingEntity)) {
                     return false;
                  }
               }
            }

            return super.attackEntityFrom(source, amount);
         }
      }
   }

   public boolean canAttackPlayer(EntityPlayer other) {
      return !this.canPlayersAttack()?false:super.canAttackPlayer(other);
   }

   private boolean canPlayersAttack() {
      return this.mcServer.isPVPEnabled();
   }

   @Nullable
   public Entity changeDimension(int dimensionIn) {
      this.invulnerableDimensionChange = true;
      if(this.dimension == 1 && dimensionIn == 1) {
         this.worldObj.removeEntity(this);
         if(!this.playerConqueredTheEnd) {
            this.playerConqueredTheEnd = true;
            if(this.hasAchievement(AchievementList.THE_END2)) {
               this.connection.sendPacket(new SPacketChangeGameState(4, 0.0F));
            } else {
               this.addStat(AchievementList.THE_END2);
               this.connection.sendPacket(new SPacketChangeGameState(4, 1.0F));
            }
         }

         return this;
      } else {
         if(this.dimension == 0 && dimensionIn == 1) {
            this.addStat(AchievementList.THE_END);
            dimensionIn = 1;
         } else {
            this.addStat(AchievementList.PORTAL);
         }

         this.mcServer.getPlayerList().changePlayerDimension(this, dimensionIn);
         this.connection.sendPacket(new SPacketEffect(1032, BlockPos.ORIGIN, 0, false));
         this.lastExperience = -1;
         this.lastHealth = -1.0F;
         this.lastFoodLevel = -1;
         return this;
      }
   }

   public boolean isSpectatedByPlayer(EntityPlayerMP player) {
      return player.isSpectator()?this.getSpectatingEntity() == this:(this.isSpectator()?false:super.isSpectatedByPlayer(player));
   }

   private void sendTileEntityUpdate(TileEntity p_147097_1_) {
      if(p_147097_1_ != null) {
         SPacketUpdateTileEntity spacketupdatetileentity = p_147097_1_.func_189518_D_();
         if(spacketupdatetileentity != null) {
            this.connection.sendPacket(spacketupdatetileentity);
         }
      }
   }

   public void onItemPickup(Entity entityIn, int quantity) {
      super.onItemPickup(entityIn, quantity);
      this.openContainer.detectAndSendChanges();
   }

   public EntityPlayer.SleepResult trySleep(BlockPos bedLocation) {
      EntityPlayer.SleepResult entityplayer$sleepresult = super.trySleep(bedLocation);
      if(entityplayer$sleepresult == EntityPlayer.SleepResult.OK) {
         this.addStat(StatList.SLEEP_IN_BED);
         Packet<?> packet = new SPacketUseBed(this, bedLocation);
         this.getServerWorld().getEntityTracker().sendToAllTrackingEntity(this, packet);
         this.connection.setPlayerLocation(this.posX, this.posY, this.posZ, this.rotationYaw, this.rotationPitch);
         this.connection.sendPacket(packet);
      }

      return entityplayer$sleepresult;
   }

   public void wakeUpPlayer(boolean immediately, boolean updateWorldFlag, boolean setSpawn) {
      if(this.isPlayerSleeping()) {
         this.getServerWorld().getEntityTracker().sendToTrackingAndSelf(this, new SPacketAnimation(this, 2));
      }

      super.wakeUpPlayer(immediately, updateWorldFlag, setSpawn);
      if(this.connection != null) {
         this.connection.setPlayerLocation(this.posX, this.posY, this.posZ, this.rotationYaw, this.rotationPitch);
      }
   }

   public boolean startRiding(Entity entityIn, boolean force) {
      Entity entity = this.getRidingEntity();
      if(!super.startRiding(entityIn, force)) {
         return false;
      } else {
         Entity entity1 = this.getRidingEntity();
         if(entity1 != entity && this.connection != null) {
            this.connection.setPlayerLocation(this.posX, this.posY, this.posZ, this.rotationYaw, this.rotationPitch);
         }

         return true;
      }
   }

   public void dismountRidingEntity() {
      Entity entity = this.getRidingEntity();
      super.dismountRidingEntity();
      Entity entity1 = this.getRidingEntity();
      if(entity1 != entity && this.connection != null) {
         this.connection.setPlayerLocation(this.posX, this.posY, this.posZ, this.rotationYaw, this.rotationPitch);
      }
   }

   public boolean isEntityInvulnerable(DamageSource source) {
      return super.isEntityInvulnerable(source) || this.isInvulnerableDimensionChange();
   }

   protected void updateFallState(double y, boolean onGroundIn, IBlockState state, BlockPos pos) {
   }

   protected void frostWalk(BlockPos pos) {
      if(!this.isSpectator()) {
         super.frostWalk(pos);
      }
   }

   public void handleFalling(double y, boolean onGroundIn) {
      int i = MathHelper.floor_double(this.posX);
      int j = MathHelper.floor_double(this.posY - 0.20000000298023224D);
      int k = MathHelper.floor_double(this.posZ);
      BlockPos blockpos = new BlockPos(i, j, k);
      IBlockState iblockstate = this.worldObj.getBlockState(blockpos);
      if(iblockstate.getMaterial() == Material.AIR) {
         BlockPos blockpos1 = blockpos.down();
         IBlockState iblockstate1 = this.worldObj.getBlockState(blockpos1);
         Block block = iblockstate1.getBlock();
         if(block instanceof BlockFence || block instanceof BlockWall || block instanceof BlockFenceGate) {
            blockpos = blockpos1;
            iblockstate = iblockstate1;
         }
      }

      super.updateFallState(y, onGroundIn, iblockstate, blockpos);
   }

   public void openEditSign(TileEntitySign signTile) {
      signTile.setPlayer(this);
      this.connection.sendPacket(new SPacketSignEditorOpen(signTile.getPos()));
   }

   private void getNextWindowId() {
      this.currentWindowId = this.currentWindowId % 100 + 1;
   }

   public void displayGui(IInteractionObject guiOwner) {
      if(guiOwner instanceof ILootContainer && ((ILootContainer)guiOwner).getLootTable() != null && this.isSpectator()) {
         this.addChatMessage((new TextComponentTranslation("container.spectatorCantOpen", new Object[0])).setStyle((new Style()).setColor(TextFormatting.RED)));
      } else {
         this.getNextWindowId();
         this.connection.sendPacket(new SPacketOpenWindow(this.currentWindowId, guiOwner.getGuiID(), guiOwner.getDisplayName()));
         this.openContainer = guiOwner.createContainer(this.inventory, this);
         this.openContainer.windowId = this.currentWindowId;
         this.openContainer.addListener(this);
      }
   }

   public void displayGUIChest(IInventory chestInventory) {
      if(chestInventory instanceof ILootContainer && ((ILootContainer)chestInventory).getLootTable() != null && this.isSpectator()) {
         this.addChatMessage((new TextComponentTranslation("container.spectatorCantOpen", new Object[0])).setStyle((new Style()).setColor(TextFormatting.RED)));
      } else {
         if(this.openContainer != this.inventoryContainer) {
            this.closeScreen();
         }

         if(chestInventory instanceof ILockableContainer) {
            ILockableContainer ilockablecontainer = (ILockableContainer)chestInventory;
            if(ilockablecontainer.isLocked() && !this.canOpen(ilockablecontainer.getLockCode()) && !this.isSpectator()) {
               this.connection.sendPacket(new SPacketChat(new TextComponentTranslation("container.isLocked", new Object[]{chestInventory.getDisplayName()}), (byte)2));
               this.connection.sendPacket(new SPacketSoundEffect(SoundEvents.BLOCK_CHEST_LOCKED, SoundCategory.BLOCKS, this.posX, this.posY, this.posZ, 1.0F, 1.0F));
               return;
            }
         }

         this.getNextWindowId();
         if(chestInventory instanceof IInteractionObject) {
            this.connection.sendPacket(new SPacketOpenWindow(this.currentWindowId, ((IInteractionObject)chestInventory).getGuiID(), chestInventory.getDisplayName(), chestInventory.getSizeInventory()));
            this.openContainer = ((IInteractionObject)chestInventory).createContainer(this.inventory, this);
         } else {
            this.connection.sendPacket(new SPacketOpenWindow(this.currentWindowId, "minecraft:container", chestInventory.getDisplayName(), chestInventory.getSizeInventory()));
            this.openContainer = new ContainerChest(this.inventory, chestInventory, this);
         }

         this.openContainer.windowId = this.currentWindowId;
         this.openContainer.addListener(this);
      }
   }

   public void displayVillagerTradeGui(IMerchant villager) {
      this.getNextWindowId();
      this.openContainer = new ContainerMerchant(this.inventory, villager, this.worldObj);
      this.openContainer.windowId = this.currentWindowId;
      this.openContainer.addListener(this);
      IInventory iinventory = ((ContainerMerchant)this.openContainer).getMerchantInventory();
      ITextComponent itextcomponent = villager.getDisplayName();
      this.connection.sendPacket(new SPacketOpenWindow(this.currentWindowId, "minecraft:villager", itextcomponent, iinventory.getSizeInventory()));
      MerchantRecipeList merchantrecipelist = villager.getRecipes(this);
      if(merchantrecipelist != null) {
         PacketBuffer packetbuffer = new PacketBuffer(Unpooled.buffer());
         packetbuffer.writeInt(this.currentWindowId);
         merchantrecipelist.writeToBuf(packetbuffer);
         this.connection.sendPacket(new SPacketCustomPayload("MC|TrList", packetbuffer));
      }
   }

   public void openGuiHorseInventory(EntityHorse horse, IInventory inventoryIn) {
      if(this.openContainer != this.inventoryContainer) {
         this.closeScreen();
      }

      this.getNextWindowId();
      this.connection.sendPacket(new SPacketOpenWindow(this.currentWindowId, "EntityHorse", inventoryIn.getDisplayName(), inventoryIn.getSizeInventory(), horse.getEntityId()));
      this.openContainer = new ContainerHorseInventory(this.inventory, inventoryIn, horse, this);
      this.openContainer.windowId = this.currentWindowId;
      this.openContainer.addListener(this);
   }

   public void openBook(ItemStack stack, EnumHand hand) {
      Item item = stack.getItem();
      if(item == Items.WRITTEN_BOOK) {
         PacketBuffer packetbuffer = new PacketBuffer(Unpooled.buffer());
         packetbuffer.writeEnumValue(hand);
         this.connection.sendPacket(new SPacketCustomPayload("MC|BOpen", packetbuffer));
      }
   }

   public void displayGuiCommandBlock(TileEntityCommandBlock p_184824_1_) {
      p_184824_1_.setSendToClient(true);
      this.sendTileEntityUpdate(p_184824_1_);
   }

   public void sendSlotContents(Container containerToSend, int slotInd, ItemStack stack) {
      if(!(containerToSend.getSlot(slotInd) instanceof SlotCrafting)) {
         if(!this.isChangingQuantityOnly) {
            this.connection.sendPacket(new SPacketSetSlot(containerToSend.windowId, slotInd, stack));
         }
      }
   }

   public void sendContainerToPlayer(Container containerIn) {
      this.updateCraftingInventory(containerIn, containerIn.getInventory());
   }

   public void updateCraftingInventory(Container containerToSend, List<ItemStack> itemsList) {
      this.connection.sendPacket(new SPacketWindowItems(containerToSend.windowId, itemsList));
      this.connection.sendPacket(new SPacketSetSlot(-1, -1, this.inventory.getItemStack()));
   }

   public void sendProgressBarUpdate(Container containerIn, int varToUpdate, int newValue) {
      this.connection.sendPacket(new SPacketWindowProperty(containerIn.windowId, varToUpdate, newValue));
   }

   public void sendAllWindowProperties(Container containerIn, IInventory inventory) {
      for(int i = 0; i < inventory.getFieldCount(); ++i) {
         this.connection.sendPacket(new SPacketWindowProperty(containerIn.windowId, i, inventory.getField(i)));
      }
   }

   public void closeScreen() {
      this.connection.sendPacket(new SPacketCloseWindow(this.openContainer.windowId));
      this.closeContainer();
   }

   public void updateHeldItem() {
      if(!this.isChangingQuantityOnly) {
         this.connection.sendPacket(new SPacketSetSlot(-1, -1, this.inventory.getItemStack()));
      }
   }

   public void closeContainer() {
      this.openContainer.onContainerClosed(this);
      this.openContainer = this.inventoryContainer;
   }

   public void setEntityActionState(float strafe, float forward, boolean jumping, boolean sneaking) {
      if(this.isRiding()) {
         if(strafe >= -1.0F && strafe <= 1.0F) {
            this.moveStrafing = strafe;
         }

         if(forward >= -1.0F && forward <= 1.0F) {
            this.moveForward = forward;
         }

         this.isJumping = jumping;
         this.setSneaking(sneaking);
      }
   }

   public boolean hasAchievement(Achievement achievementIn) {
      return this.statsFile.hasAchievementUnlocked(achievementIn);
   }

   public void addStat(StatBase stat, int amount) {
      if(stat != null) {
         this.statsFile.increaseStat(this, stat, amount);

         for(ScoreObjective scoreobjective : this.getWorldScoreboard().getObjectivesFromCriteria(stat.getCriteria())) {
            this.getWorldScoreboard().getOrCreateScore(this.getName(), scoreobjective).increaseScore(amount);
         }

         if(this.statsFile.hasUnsentAchievement()) {
            this.statsFile.sendStats(this);
         }
      }
   }

   public void takeStat(StatBase stat) {
      if(stat != null) {
         this.statsFile.unlockAchievement(this, stat, 0);

         for(ScoreObjective scoreobjective : this.getWorldScoreboard().getObjectivesFromCriteria(stat.getCriteria())) {
            this.getWorldScoreboard().getOrCreateScore(this.getName(), scoreobjective).setScorePoints(0);
         }

         if(this.statsFile.hasUnsentAchievement()) {
            this.statsFile.sendStats(this);
         }
      }
   }

   public void mountEntityAndWakeUp() {
      this.removePassengers();
      if(this.sleeping) {
         this.wakeUpPlayer(true, false, false);
      }
   }

   public void setPlayerHealthUpdated() {
      this.lastHealth = -1.0E8F;
   }

   public void addChatComponentMessage(ITextComponent chatComponent) {
      this.connection.sendPacket(new SPacketChat(chatComponent));
   }

   protected void onItemUseFinish() {
      if(this.activeItemStack != null && this.isHandActive()) {
         this.connection.sendPacket(new SPacketEntityStatus(this, (byte)9));
         super.onItemUseFinish();
      }
   }

   public void clonePlayer(EntityPlayer oldPlayer, boolean respawnFromEnd) {
      super.clonePlayer(oldPlayer, respawnFromEnd);
      this.lastExperience = -1;
      this.lastHealth = -1.0F;
      this.lastFoodLevel = -1;
      this.entityRemoveQueue.addAll(((EntityPlayerMP)oldPlayer).entityRemoveQueue);
   }

   protected void onNewPotionEffect(PotionEffect id) {
      super.onNewPotionEffect(id);
      this.connection.sendPacket(new SPacketEntityEffect(this.getEntityId(), id));
   }

   protected void onChangedPotionEffect(PotionEffect id, boolean p_70695_2_) {
      super.onChangedPotionEffect(id, p_70695_2_);
      this.connection.sendPacket(new SPacketEntityEffect(this.getEntityId(), id));
   }

   protected void onFinishedPotionEffect(PotionEffect effect) {
      super.onFinishedPotionEffect(effect);
      this.connection.sendPacket(new SPacketRemoveEntityEffect(this.getEntityId(), effect.getPotion()));
   }

   public void setPositionAndUpdate(double x, double y, double z) {
      this.connection.setPlayerLocation(x, y, z, this.rotationYaw, this.rotationPitch);
   }

   public void onCriticalHit(Entity entityHit) {
      this.getServerWorld().getEntityTracker().sendToTrackingAndSelf(this, new SPacketAnimation(entityHit, 4));
   }

   public void onEnchantmentCritical(Entity entityHit) {
      this.getServerWorld().getEntityTracker().sendToTrackingAndSelf(this, new SPacketAnimation(entityHit, 5));
   }

   public void sendPlayerAbilities() {
      if(this.connection != null) {
         this.connection.sendPacket(new SPacketPlayerAbilities(this.capabilities));
         this.updatePotionMetadata();
      }
   }

   public WorldServer getServerWorld() {
      return (WorldServer)this.worldObj;
   }

   public void setGameType(GameType gameType) {
      this.interactionManager.setGameType(gameType);
      this.connection.sendPacket(new SPacketChangeGameState(3, (float)gameType.getID()));
      if(gameType == GameType.SPECTATOR) {
         this.dismountRidingEntity();
      } else {
         this.setSpectatingEntity(this);
      }

      this.sendPlayerAbilities();
      this.markPotionsDirty();
   }

   public boolean isSpectator() {
      return this.interactionManager.getGameType() == GameType.SPECTATOR;
   }

   public boolean isCreative() {
      return this.interactionManager.getGameType() == GameType.CREATIVE;
   }

   public void addChatMessage(ITextComponent component) {
      this.connection.sendPacket(new SPacketChat(component));
   }

   public boolean canCommandSenderUseCommand(int permLevel, String commandName) {
      if("seed".equals(commandName) && !this.mcServer.isDedicatedServer()) {
         return true;
      } else if(!"tell".equals(commandName) && !"help".equals(commandName) && !"me".equals(commandName) && !"trigger".equals(commandName)) {
         if(this.mcServer.getPlayerList().canSendCommands(this.getGameProfile())) {
            UserListOpsEntry userlistopsentry = (UserListOpsEntry)this.mcServer.getPlayerList().getOppedPlayers().getEntry(this.getGameProfile());
            return userlistopsentry != null?userlistopsentry.getPermissionLevel() >= permLevel:this.mcServer.getOpPermissionLevel() >= permLevel;
         } else {
            return false;
         }
      } else {
         return true;
      }
   }

   public String getPlayerIP() {
      String s = this.connection.netManager.getRemoteAddress().toString();
      s = s.substring(s.indexOf("/") + 1);
      s = s.substring(0, s.indexOf(":"));
      return s;
   }

   public void handleClientSettings(CPacketClientSettings packetIn) {
      this.language = packetIn.getLang();
      this.chatVisibility = packetIn.getChatVisibility();
      this.chatColours = packetIn.isColorsEnabled();
      this.getDataManager().set(PLAYER_MODEL_FLAG, Byte.valueOf((byte)packetIn.getModelPartFlags()));
      this.getDataManager().set(MAIN_HAND, Byte.valueOf((byte)(packetIn.getMainHand() == EnumHandSide.LEFT?0:1)));
   }

   public EntityPlayer.EnumChatVisibility getChatVisibility() {
      return this.chatVisibility;
   }

   public void loadResourcePack(String url, String hash) {
      this.connection.sendPacket(new SPacketResourcePackSend(url, hash));
   }

   public BlockPos getPosition() {
      return new BlockPos(this.posX, this.posY + 0.5D, this.posZ);
   }

   public void markPlayerActive() {
      this.playerLastActiveTime = MinecraftServer.getCurrentTimeMillis();
   }

   public StatisticsManagerServer getStatFile() {
      return this.statsFile;
   }

   public void removeEntity(Entity entityIn) {
      if(entityIn instanceof EntityPlayer) {
         this.connection.sendPacket(new SPacketDestroyEntities(new int[]{entityIn.getEntityId()}));
      } else {
         this.entityRemoveQueue.add(Integer.valueOf(entityIn.getEntityId()));
      }
   }

   public void addEntity(Entity entityIn) {
      this.entityRemoveQueue.remove(Integer.valueOf(entityIn.getEntityId()));
   }

   protected void updatePotionMetadata() {
      if(this.isSpectator()) {
         this.resetPotionEffectMetadata();
         this.setInvisible(true);
      } else {
         super.updatePotionMetadata();
      }

      this.getServerWorld().getEntityTracker().updateVisibility(this);
   }

   public Entity getSpectatingEntity() {
      return (Entity)(this.spectatingEntity == null?this:this.spectatingEntity);
   }

   public void setSpectatingEntity(Entity entityToSpectate) {
      Entity entity = this.getSpectatingEntity();
      this.spectatingEntity = (Entity)(entityToSpectate == null?this:entityToSpectate);
      if(entity != this.spectatingEntity) {
         this.connection.sendPacket(new SPacketCamera(this.spectatingEntity));
         this.setPositionAndUpdate(this.spectatingEntity.posX, this.spectatingEntity.posY, this.spectatingEntity.posZ);
      }
   }

   protected void decrementTimeUntilPortal() {
      if(this.timeUntilPortal > 0 && !this.invulnerableDimensionChange) {
         --this.timeUntilPortal;
      }
   }

   public void attackTargetEntityWithCurrentItem(Entity targetEntity) {
      if(this.interactionManager.getGameType() == GameType.SPECTATOR) {
         this.setSpectatingEntity(targetEntity);
      } else {
         super.attackTargetEntityWithCurrentItem(targetEntity);
      }
   }

   public long getLastActiveTime() {
      return this.playerLastActiveTime;
   }

   @Nullable
   public ITextComponent getTabListDisplayName() {
      return null;
   }

   public void swingArm(EnumHand hand) {
      super.swingArm(hand);
      this.resetCooldown();
   }

   public boolean isInvulnerableDimensionChange() {
      return this.invulnerableDimensionChange;
   }

   public void clearInvulnerableDimensionChange() {
      this.invulnerableDimensionChange = false;
   }

   public void setElytraFlying() {
      this.setFlag(7, true);
   }

   public void clearElytraFlying() {
      this.setFlag(7, true);
      this.setFlag(7, false);
   }
   
   public void queueChunkPacket(SPacketChunkData packet) {
	   chunkQueue.add(packet);
   }
}
